<?php

namespace App\Domain\Business;

use App\Domain\Wallet\WalletTransferService;
use App\Models\PaymentLink;
use App\Models\PaymentLinkPayment;
use App\Models\User;
use App\Services\FeatureGateService;
use App\Support\Security\PublicIdGenerator;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PaymentLinkService
{
    public function __construct(
        private readonly PublicIdGenerator $idGenerator,
        private readonly WalletTransferService $walletTransferService,
        private readonly FeatureGateService $featureGate,
    ) {
    }

    public function createLink(User $merchant, array $payload): PaymentLink
    {
        if (! $this->featureGate->isEnabled('business.payment_links', $merchant)) {
            throw ValidationException::withMessages(['feature' => 'Payment links are disabled.']);
        }

        if (! $merchant->businessProfile?->is_business_enabled) {
            throw ValidationException::withMessages(['business' => 'Business features are disabled for this account.']);
        }

        return PaymentLink::query()->create([
            'merchant_user_id' => $merchant->id,
            'code' => $this->idGenerator->generatePaymentCode(),
            'title' => $payload['title'],
            'description' => $payload['description'] ?? null,
            'fixed_amount' => $payload['fixed_amount'] ?? null,
            'allow_custom_amount' => (bool) ($payload['allow_custom_amount'] ?? false),
            'expires_at' => $payload['expires_at'] ?? null,
            'single_use' => (bool) ($payload['single_use'] ?? true),
            'status' => 'active',
        ]);
    }

    public function payLink(PaymentLink $link, User $payer, int $amount, string $idempotencyKey, ?string $totpCode = null): PaymentLinkPayment
    {
        if ($link->status !== 'active') {
            throw ValidationException::withMessages(['payment_link' => 'Link is not active.']);
        }

        if ($link->expires_at && now()->greaterThan($link->expires_at)) {
            $link->update(['status' => 'expired']);
            throw ValidationException::withMessages(['payment_link' => 'Link is expired.']);
        }

        if (! $link->allow_custom_amount && $link->fixed_amount !== $amount) {
            throw ValidationException::withMessages(['amount' => 'Amount must match link amount.']);
        }

        if ($link->allow_custom_amount && $amount <= 0) {
            throw ValidationException::withMessages(['amount' => 'Custom amount must be positive.']);
        }

        return DB::transaction(function () use ($link, $payer, $amount, $idempotencyKey, $totpCode) {
            $link = PaymentLink::query()->lockForUpdate()->findOrFail($link->id);
            if ($link->single_use && PaymentLinkPayment::query()->where('payment_link_id', $link->id)->where('status', 'completed')->exists()) {
                throw ValidationException::withMessages(['payment_link' => 'This payment link is already paid.']);
            }

            $merchant = User::query()->findOrFail($link->merchant_user_id);
            $transfer = $this->walletTransferService->transfer(
                sender: $payer,
                receiver: $merchant,
                amount: $amount,
                idempotencyKey: $idempotencyKey,
                note: 'Payment link: '.$link->code,
                totpCode: $totpCode,
            );

            $payment = PaymentLinkPayment::query()->create([
                'payment_link_id' => $link->id,
                'payer_user_id' => $payer->id,
                'amount' => $amount,
                'status' => 'completed',
                'transfer_id' => $transfer->id,
            ]);

            if ($link->single_use) {
                $link->status = 'paid';
                $link->save();
            }

            return $payment;
        });
    }
}
