<?php

namespace App\Http\Controllers\Api\V1;

use App\Domain\Business\PaymentLinkService;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\PayPaymentLinkRequest;
use App\Http\Requests\Api\StorePaymentLinkRequest;
use App\Models\PaymentLink;
use Illuminate\Http\JsonResponse;

class PaymentLinkController extends Controller
{
    public function __construct(private readonly PaymentLinkService $paymentLinkService)
    {
    }

    public function store(StorePaymentLinkRequest $request): JsonResponse
    {
        $data = $request->validated();
        if (! $data['confirm']) {
            return response()->json(['message' => 'Action not confirmed.'], 422);
        }

        $link = $this->paymentLinkService->createLink($request->user(), $data);

        return response()->json(['code' => $link->code, 'status' => $link->status], 201);
    }

    public function show(string $code): JsonResponse
    {
        $link = PaymentLink::query()->where('code', $code)->firstOrFail();

        return response()->json([
            'code' => $link->code,
            'title' => $link->title,
            'description' => $link->description,
            'fixed_amount' => $link->fixed_amount,
            'allow_custom_amount' => $link->allow_custom_amount,
            'status' => $link->status,
            'expires_at' => $link->expires_at,
            'merchant' => [
                'business_name' => $link->merchant->businessProfile?->business_name,
                'public_id' => $link->merchant->public_id,
            ],
        ]);
    }

    public function pay(PayPaymentLinkRequest $request, string $code): JsonResponse
    {
        $data = $request->validated();
        if (! $data['confirm']) {
            return response()->json(['message' => 'Action not confirmed.'], 422);
        }

        $link = PaymentLink::query()->where('code', $code)->firstOrFail();

        $payment = $this->paymentLinkService->payLink(
            link: $link,
            payer: $request->user(),
            amount: (int) $data['amount'],
            idempotencyKey: $data['idempotency_key'],
            totpCode: $data['totp_code'] ?? null,
        );

        return response()->json(['payment_id' => $payment->id, 'status' => $payment->status]);
    }
}
