<?php

namespace App\Http\Controllers\Web;

use App\Domain\Wallet\WalletTransferService;
use App\Http\Controllers\Controller;
use App\Http\Requests\Wallet\StoreTransferRequest;
use App\Models\LedgerEntry;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class WalletController extends Controller
{
    public function __construct(private readonly WalletTransferService $walletTransferService)
    {
    }

    public function dashboard(): View
    {
        $user = auth()->user();

        $transactions = LedgerEntry::query()
            ->whereHas('wallet', fn ($q) => $q->where('user_id', $user->id))
            ->latest()
            ->limit(20)
            ->get();

        return view('wallet.dashboard', [
            'user' => $user,
            'wallet' => $user->wallet,
            'transactions' => $transactions,
        ]);
    }

    public function send(StoreTransferRequest $request): RedirectResponse
    {
        $data = $request->validated();

        if (! $data['confirm']) {
            return back()->withErrors(['confirm' => 'Action must be confirmed.']);
        }

        $receiver = User::query()->where('public_id', $data['receiver_public_id'])->firstOrFail();

        $this->walletTransferService->transfer(
            sender: $request->user(),
            receiver: $receiver,
            amount: (int) $data['amount'],
            idempotencyKey: $data['idempotency_key'],
            note: $data['note'] ?? null,
            totpCode: $data['totp_code'] ?? null,
        );

        return redirect()->route('wallet.dashboard')->with('status', 'Transfer completed.');
    }
}
