<?php

declare(strict_types=1);

use App\Models\SystemSetting;
use App\Models\User;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Hash;

$basePath = dirname(__DIR__, 2);
$lockFile = $basePath.'/storage/app/installed.lock';
$envPath = $basePath.'/.env';
$envExample = $basePath.'/.env.example';

if (file_exists($lockFile)) {
    echo '<h2>Application is already installed.</h2>';
    echo '<p>Delete storage/app/installed.lock only if you intentionally want to reinstall.</p>';
    exit;
}

$errors = [];
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = fn (string $key, string $default = ''): string => trim((string) ($_POST[$key] ?? $default));

    $dbDriver = $input('db_driver', 'mysql');
    $appName = $input('app_name', 'Secure Wallet');
    $appUrl = $input('app_url', 'http://localhost');
    $copyright = $input('copyright');

    $dbHost = $input('db_host');
    $dbPort = $input('db_port', $dbDriver === 'pgsql' ? '5432' : '3306');
    $dbName = $input('db_name');
    $dbUser = $input('db_user');
    $dbPass = $input('db_pass');
    $dbUrl = $input('db_url');
    $dbSslMode = $input('db_sslmode', $dbDriver === 'pgsql' ? 'require' : 'prefer');

    $adminUsername = $input('admin_username');
    $adminEmail = $input('admin_email');
    $adminPassword = $input('admin_password');

    if (! in_array($dbDriver, ['mysql', 'pgsql'], true)) {
        $errors[] = 'Unsupported DB driver.';
    }

    // DB URL should be used for PostgreSQL/Supabase only.
    if ($dbDriver !== 'pgsql') {
        $dbUrl = '';
    }

    $required = [
        'App Name' => $appName,
        'App URL' => $appUrl,
        'Admin Username' => $adminUsername,
        'Admin Email' => $adminEmail,
        'Admin Password' => $adminPassword,
    ];

    if ($dbUrl === '') {
        $required['DB Host'] = $dbHost;
        $required['DB Name'] = $dbName;
        $required['DB User'] = $dbUser;
        $required['DB Password'] = $dbPass;
    }

    foreach ($required as $label => $value) {
        if ($value === '') {
            $errors[] = "$label is required.";
        }
    }

    if (! filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Admin email format is invalid.';
    }

    if (strlen($adminPassword) < 8) {
        $errors[] = 'Admin password must be at least 8 characters.';
    }

    if ($dbDriver === 'pgsql' && $dbUrl !== '') {
        if (str_contains($dbUrl, '[PASSWORD]')) {
            $errors[] = 'DATABASE_URL still contains [PASSWORD]. Replace it with your real password.';
        }

        if (parse_url($dbUrl) === false) {
            $errors[] = 'DATABASE_URL is malformed. If password contains #, use %23 instead.';
        }
    }

    if (! file_exists($envExample)) {
        $errors[] = '.env.example not found.';
    }

    if (! is_writable($basePath)) {
        $errors[] = 'Project root is not writable.';
    }

    if (! is_writable($basePath.'/storage')) {
        $errors[] = 'Storage directory is not writable.';
    }

    if ($errors === []) {
        try {
            $envContent = file_get_contents($envExample);
            if ($envContent === false) {
                throw new RuntimeException('Unable to read .env.example');
            }

            $appKey = 'base64:'.base64_encode(random_bytes(32));

            $replacements = [
                'APP_NAME' => '"'.str_replace('"', '\\"', $appName).'"',
                'APP_ENV' => 'production',
                'APP_KEY' => $appKey,
                'APP_DEBUG' => 'false',
                'APP_URL' => $appUrl,
                'APP_INSTALLED' => 'true',
                'APP_COPYRIGHT' => '"'.str_replace('"', '\\"', $copyright).'"',
                'DB_CONNECTION' => $dbDriver,
                'DB_URL' => $dbUrl !== '' ? '"'.str_replace('"', '\\"', $dbUrl).'"' : '',
                'DB_HOST' => $dbHost,
                'DB_PORT' => $dbPort,
                'DB_DATABASE' => $dbName,
                'DB_USERNAME' => $dbUser,
                'DB_PASSWORD' => '"'.str_replace('"', '\\"', $dbPass).'"',
                'DB_SSLMODE' => $dbSslMode,
                'CACHE_STORE' => 'file',
            ];

            $upsertEnvVar = static function (string $content, string $key, string $value): string {
                $pattern = '/^(?:#\s*)?'.preg_quote($key, '/').'=.*$/m';
                $line = $key.'='.$value;

                if (preg_match($pattern, $content) === 1) {
                    return preg_replace($pattern, $line, $content, 1) ?? $content;
                }

                return rtrim($content).PHP_EOL.$line.PHP_EOL;
            };

            foreach ($replacements as $key => $value) {
                $envContent = $upsertEnvVar($envContent, $key, $value);
            }

            if (file_put_contents($envPath, $envContent) === false) {
                throw new RuntimeException('Unable to write .env file');
            }

            foreach ([
                $basePath.'/bootstrap/cache/config.php',
                $basePath.'/bootstrap/cache/routes-v7.php',
                $basePath.'/bootstrap/cache/packages.php',
                $basePath.'/bootstrap/cache/services.php',
            ] as $cacheFile) {
                if (file_exists($cacheFile)) {
                    @unlink($cacheFile);
                }
            }

            require $basePath.'/vendor/autoload.php';
            $app = require $basePath.'/bootstrap/app.php';
            $kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
            $kernel->bootstrap();

            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
            Artisan::call('config:clear');

            $admin = User::query()->firstOrCreate(
                ['email' => $adminEmail],
                [
                    'username' => $adminUsername,
                    'password' => Hash::make($adminPassword),
                    'status' => 'active',
                    'email_verified_at' => now(),
                ]
            );
            $admin->assignRole('super_admin');

            SystemSetting::setValue('system.name', $appName);
            SystemSetting::setValue('system.copyright', $copyright);

            if (! is_dir(dirname($lockFile))) {
                mkdir(dirname($lockFile), 0775, true);
            }

            file_put_contents($lockFile, 'installed_at='.date(DATE_ATOM));

            $success = 'Installation completed successfully. Delete /public/install folder for extra safety.';
        } catch (Throwable $e) {
            $errors[] = 'Installation failed: '.$e->getMessage();
        }
    }
}

function e_html(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Wallet Installer</title>
  <style>
    body{font-family:Arial,sans-serif;background:#f4f6f8;margin:0;padding:20px}
    .card{max-width:980px;margin:auto;background:#fff;border-radius:12px;padding:20px;box-shadow:0 8px 24px rgba(0,0,0,.08)}
    .grid{display:grid;grid-template-columns:1fr 1fr;gap:12px}
    label{display:block;font-size:13px;color:#444;margin-bottom:4px}
    input,select{width:100%;padding:10px;border:1px solid #d1d5db;border-radius:8px}
    h2{margin-top:0}
    .full{grid-column:1 / -1}
    .btn{padding:11px 16px;background:#0369a1;color:#fff;border:0;border-radius:8px;cursor:pointer}
    .err{background:#fee2e2;color:#991b1b;padding:10px;border-radius:8px;margin-bottom:10px}
    .ok{background:#dcfce7;color:#166534;padding:10px;border-radius:8px;margin-bottom:10px}
    @media(max-width:900px){.grid{grid-template-columns:1fr}}
  </style>
</head>
<body>
<div class="card">
  <h2>Secure Wallet Installer</h2>
  <p>Fill all required fields, then install. Supports MySQL (phpMyAdmin hosting) and Supabase PostgreSQL.</p>

  <?php if ($success): ?>
    <div class="ok"><?= e_html($success) ?></div>
  <?php endif; ?>

  <?php foreach ($errors as $error): ?>
    <div class="err"><?= e_html($error) ?></div>
  <?php endforeach; ?>

  <form method="post">
    <div class="grid">
      <div>
        <label>System Name</label>
        <input name="app_name" value="<?= e_html($_POST['app_name'] ?? 'Secure Wallet') ?>" required>
      </div>
      <div>
        <label>System URL</label>
        <input name="app_url" value="<?= e_html($_POST['app_url'] ?? 'http://localhost') ?>" required>
      </div>
      <div class="full">
        <label>Copyright</label>
        <input name="copyright" value="<?= e_html($_POST['copyright'] ?? '') ?>">
      </div>

      <div>
        <label>Database Driver</label>
        <select name="db_driver">
          <option value="mysql" <?= (($_POST['db_driver'] ?? 'mysql') === 'mysql') ? 'selected' : '' ?>>MySQL (shared hosting / phpMyAdmin)</option>
          <option value="pgsql" <?= (($_POST['db_driver'] ?? '') === 'pgsql') ? 'selected' : '' ?>>PostgreSQL (Supabase)</option>
        </select>
      </div>
      <div>
        <label>DB SSL Mode</label>
        <input name="db_sslmode" value="<?= e_html($_POST['db_sslmode'] ?? 'prefer') ?>">
      </div>
      <div class="full">
        <label>DATABASE_URL (Optional - recommended for Supabase)</label>
        <input name="db_url" value="<?= e_html($_POST['db_url'] ?? '') ?>" placeholder="postgresql://postgres:[PASSWORD]@db.<project-ref>.supabase.co:5432/postgres">
      </div>
      <div>
        <label>DB Host</label>
        <input name="db_host" value="<?= e_html($_POST['db_host'] ?? '') ?>">
      </div>
      <div>
        <label>DB Port</label>
        <input name="db_port" value="<?= e_html($_POST['db_port'] ?? '3306') ?>">
      </div>
      <div>
        <label>DB Name</label>
        <input name="db_name" value="<?= e_html($_POST['db_name'] ?? '') ?>">
      </div>
      <div>
        <label>DB Username</label>
        <input name="db_user" value="<?= e_html($_POST['db_user'] ?? '') ?>">
      </div>
      <div class="full">
        <label>DB Password</label>
        <input type="password" name="db_pass" value="<?= e_html($_POST['db_pass'] ?? '') ?>">
      </div>

      <div>
        <label>Admin Username</label>
        <input name="admin_username" value="<?= e_html($_POST['admin_username'] ?? '') ?>" required>
      </div>
      <div>
        <label>Admin Email</label>
        <input name="admin_email" value="<?= e_html($_POST['admin_email'] ?? '') ?>" required>
      </div>
      <div class="full">
        <label>Admin Password</label>
        <input type="password" name="admin_password" required>
      </div>

      <div class="full">
        <button class="btn" type="submit">Install Now</button>
      </div>
    </div>
  </form>
</div>
</body>
</html>
